'use client'

import Link from 'next/link'
import Image from 'next/image'
import { useState, useEffect } from 'react'
import { Button } from '@/components/ui/button'
import { Sheet, SheetContent, SheetTrigger } from '@/components/ui/sheet'
import { NavigationMenu, NavigationMenuContent, NavigationMenuItem, NavigationMenuLink, NavigationMenuList, NavigationMenuTrigger } from '@/components/ui/navigation-menu'
import { Menu, Phone, Mail, MapPin } from 'lucide-react'

interface CompanyData {
  companyName: string
  logo?: string
  phoneNumber: string
  email: string
  address: string
}

export default function Header() {
  const [isOpen, setIsOpen] = useState(false)
  const [companyData, setCompanyData] = useState<CompanyData | null>(null)

  useEffect(() => {
    const fetchCompanyData = async () => {
      try {
        const response = await fetch('/api/admin/company')
        if (response.ok) {
          const data = await response.json()
          if (data.length > 0) {
            setCompanyData(data[0])
          }
        }
      } catch (error) {
        console.error('Failed to fetch company data:', error)
      }
    }

    fetchCompanyData()
  }, [])

  const navigation = [
    { name: 'Home', href: '/' },
    { 
      name: 'Our Products', 
      href: '/products',
      submenu: [
        { name: 'Office Porta Cabin', href: '/products/office-cabin' },
        { name: 'Security Cabin', href: '/products/security-cabin' },
        { name: 'Toilet Units', href: '/products/toilet-units' },
        { name: 'Custom Solutions', href: '/products/custom' },
      ]
    },
    { name: 'Our Work', href: '/work' },
    { name: 'About Us', href: '/about' },
    { name: 'Contact', href: '/contact' },
  ]

  return (
    <header className="bg-white shadow-sm border-b sticky top-0 z-50">
      {/* Top Bar */}
      <div className="bg-blue-800 text-white py-2 px-4 text-sm">
        <div className="container mx-auto flex justify-between items-center">
          <div className="flex items-center space-x-4">
            <div className="flex items-center space-x-2">
              <Phone className="h-4 w-4" />
              <span>{companyData?.phoneNumber || '+971 58 201 2073'}</span>
            </div>
            <div className="flex items-center space-x-2">
              <Mail className="h-4 w-4" />
              <span>{companyData?.email || 'sales@saamcabins.com'}</span>
            </div>
          </div>
          <div className="flex items-center space-x-2">
            <MapPin className="h-4 w-4" />
            <span>{companyData?.address || 'P2 Hamriyah, Sharjah, UAE'}</span>
          </div>
        </div>
      </div>

      {/* Main Navigation */}
      <div className="container mx-auto px-4 py-4">
        <div className="flex justify-between items-center">
          {/* Logo */}
          <Link href="/" className="flex items-center space-x-3">
            {companyData?.logo ? (
              <div className="flex items-center space-x-3">
                <Image
                  src={companyData.logo}
                  alt={companyData.companyName || 'SAAM Cabins'}
                  width={120}
                  height={60}
                  className="h-12 w-auto object-contain"
                />
                <div className="text-xl font-bold text-blue-800 hidden sm:block">
                  {companyData.companyName || 'SAAM CABINS'}
                </div>
              </div>
            ) : (
              <>
                <div className="bg-blue-800 text-white px-3 py-2 rounded-lg font-bold text-xl">
                  SAAM
                </div>
                <div className="text-xl font-bold text-blue-800">CABINS</div>
              </>
            )}
          </Link>

          {/* Desktop Navigation */}
          <NavigationMenu className="hidden lg:flex">
            <NavigationMenuList>
              {navigation.map((item) => (
                <NavigationMenuItem key={item.name}>
                  {item.submenu ? (
                    <>
                      <NavigationMenuTrigger className="text-gray-700 hover:text-green-800">
                        {item.name}
                      </NavigationMenuTrigger>
                      <NavigationMenuContent>
                        <div className="grid gap-3 p-6 w-[400px]">
                          {item.submenu.map((subItem) => (
                            <NavigationMenuLink
                              key={subItem.name}
                              href={subItem.href}
                              className="block select-none space-y-1 rounded-md p-3 leading-none no-underline outline-none transition-colors hover:bg-accent hover:text-accent-foreground focus:bg-accent focus:text-accent-foreground"
                            >
                              <div className="text-sm font-medium leading-none">{subItem.name}</div>
                            </NavigationMenuLink>
                          ))}
                        </div>
                      </NavigationMenuContent>
                    </>
                  ) : (
                    <NavigationMenuLink
                      href={item.href}
                      className="group inline-flex h-10 w-max items-center justify-center rounded-md bg-background px-4 py-2 text-sm font-medium transition-colors hover:bg-accent hover:text-accent-foreground focus:bg-accent focus:text-accent-foreground focus:outline-none disabled:pointer-events-none disabled:opacity-50 text-gray-700 hover:text-blue-800"
                    >
                      {item.name}
                    </NavigationMenuLink>
                  )}
                </NavigationMenuItem>
              ))}
            </NavigationMenuList>
          </NavigationMenu>

          {/* CTA Button and Mobile Menu */}
          <div className="flex items-center space-x-4">
            <Button className="hidden md:flex bg-blue-800 hover:bg-blue-700">
              Get Quote
            </Button>
            
            {/* Mobile menu trigger */}
            <Sheet open={isOpen} onOpenChange={setIsOpen}>
              <SheetTrigger asChild>
                <Button variant="outline" size="icon" className="lg:hidden">
                  <Menu className="h-6 w-6" />
                </Button>
              </SheetTrigger>
              <SheetContent side="right" className="w-[300px] sm:w-[400px]">
                <nav className="flex flex-col space-y-4 mt-6">
                  {navigation.map((item) => (
                    <div key={item.name}>
                      <Link
                        href={item.href}
                        className="text-lg font-medium text-gray-700 hover:text-blue-800 block py-2"
                        onClick={() => setIsOpen(false)}
                      >
                        {item.name}
                      </Link>
                      {item.submenu && (
                        <div className="ml-4 mt-2 space-y-2">
                          {item.submenu.map((subItem) => (
                            <Link
                              key={subItem.name}
                              href={subItem.href}
                              className="text-sm text-gray-600 hover:text-blue-800 block py-1"
                              onClick={() => setIsOpen(false)}
                            >
                              {subItem.name}
                            </Link>
                          ))}
                        </div>
                      )}
                    </div>
                  ))}
                  <Button className="bg-blue-800 hover:bg-blue-700 mt-6">
                    Get Quote
                  </Button>
                </nav>
              </SheetContent>
            </Sheet>
          </div>
        </div>
      </div>
    </header>
  )
}